import os
import sys
import argparse
import ipaddress
import urllib.parse
import fugue.config
import fugue.tunnel
import fugue.hub
import fugue.spoke
import fugue.args.show
import fugue.args.config
import fugue.args.tunnel
import fugue.args.interface
import fugue.args.peer
import fugue.args.hub
import fugue.args.spoke
from typing import List, cast


# TODO: what license to use? I don't want people to use this to make money. They should also have to republish
#   modifications under the same license. I guess I'd also want modifications to be open-source.
#   I think that's helpful? https://opensource.stackexchange.com/questions/4875/open-source-license-to-prevent-commercial-use  # noqa: E501
#   I think this didn't really have what I was looking for: https://man.sr.ht/license.md.

# TODO: can we run checks to ensure that a site's port is properly forwarded through the router?
# TODO: can we run checks to make sure a site's endpoint is reachable?
# TODO: can we run checks to make sure a site's has got dynamic DNS properly setup?
# TODO: should we check that the site is properly set as the LAN's default DNS, if possible?
#   I think some of this is out of scope for fugue...


def check_privileges():
    if not os.geteuid() == 0:
        print("error: you have to run as root.")
        sys.exit(3)


def check_init():
    if not os.path.isdir(fugue.config.FUGUE_ROOT):
        print("Info: {fugue.config.FUGUE_ROOT} not found, creating...")
        os.makedirs(fugue.config.FUGUE_ROOT, exist_ok=True)
        print("Info: fugue initialized")


def parse_args(init_tunnels: List[fugue.tunnel.Tunnel],
               init_hubs: List[fugue.hub.Hub],
               init_interfaces: List[fugue.interface.Interface],
               init_peers: List[fugue.peer.Peer]) -> argparse.Namespace:
    parser = argparse.ArgumentParser()

    subparsers = parser.add_subparsers(help='Fugue command', dest='command')

    # the subparser had subparsers
    fugue.args.show.set_subparser(cast(argparse.Namespace, subparsers))
    fugue.args.config.set_subparser(cast(argparse.Namespace, subparsers))
    fugue.args.interface.set_subparser(cast(argparse.Namespace, subparsers), init_interfaces)
    fugue.args.peer.set_subparser(cast(argparse.Namespace, subparsers), init_peers)
    fugue.args.tunnel.set_subparser(cast(argparse.Namespace, subparsers), init_tunnels, init_interfaces, init_peers)
    fugue.args.hub.set_subparser(cast(argparse.Namespace, subparsers))
    fugue.args.spoke.set_subparser(cast(argparse.Namespace, subparsers), init_hubs)

    # TODO: to refactor the below, just like above
    return parser.parse_args()


def main() -> None:
    check_privileges()
    check_init()
    # TODO: maybe get_state()?
    init_tunnels: List[fugue.tunnel.Tunnel] = fugue.tunnel.Tunnel.get_tunnels()
    init_hubs: List[fugue.hub.Hub] = fugue.hub.Hub.get_hubs()
    init_interfaces: List[fugue.interface.Interface] = fugue.interface.Interface.get_interfaces()
    init_peers: List[fugue.peer.Peer] = fugue.peer.Peer.get_peers()

    args = parse_args(init_tunnels, init_hubs, init_interfaces, init_peers)

    # init vars
    tunnels: List[fugue.tunnel.Tunnel]
    hubs: List[fugue.hub.Hub]
    spokes: List[fugue.spoke.Spoke]
    interfaces: List[fugue.interface.Interface]
    peers: List[fugue.peer.Peer]

    tunnel: fugue.tunnel.Tunnel
    hub: fugue.hub.Hub
    spoke: fugue.spoke.Spoke
    interface: fugue.interface.Interface
    peer: fugue.peer.Peer

    if args.command == 'show':
        interfaces = fugue.interface.Interface.get_interfaces()
        hubs = fugue.hub.Hub.get_hubs()
        spokes = fugue.spoke.Spoke.get_spokes()
        peers = fugue.peer.Peer.get_peers()
        tunnels = fugue.tunnel.Tunnel.get_tunnels()

        fugue.config.show()
        print()
        print(f'interfaces: {interfaces}')
        print()
        print(f'peers: {peers}')
        print()
        print(f'tunnels: {tunnels}')
        print()
        print(f'hubs: {hubs}')
        print()
        print(f'spokes: {spokes}')

    elif args.command == 'config':
        if args.action == 'show':
            fugue.config.show()

    elif args.command == 'hub':
        if args.action == 'gen':
            hub = fugue.hub.Hub.gen(args.name, args.listen_port, urllib.parse.urlparse(args.endpoint))
            print(hub)
            hub.persist()

        elif args.action == 'show':
            hubs = fugue.hub.Hub.get_hubs()
            print(f'hubs: {hubs}')

    elif args.command == 'spoke':
        if args.action == 'gen':
            hub = fugue.hub.Hub.from_name(args.hub_name)
            spoke = fugue.spoke.Spoke.gen(args.name, hub)
            print(spoke)
            spoke.persist()
            spoke.hub.tunnel.persist()

        elif args.action == 'show':
            if args.hub_name:
                hub = fugue.hub.Hub.from_name(args.hub_name)
                spokes = fugue.spoke.Spoke.get_spokes_for_hub(hub)

            else:
                spokes = fugue.spoke.Spoke.get_spokes()

            print(spokes)

    elif args.command == 'tunnel':
        if args.action == 'show':
            tunnels = fugue.tunnel.Tunnel.get_tunnels()

            print(f'tunnels: {tunnels}')

        elif args.action == 'new':
            interface = fugue.interface.Interface.from_name(args.interface_name)
            peers = []
            for peer_name in args.peer_names:
                peers.append(fugue.peer.Peer.from_name(peer_name))

            # TODO: how to properly handle names when those do not exist?
            tunnel_name: str = interface.name if interface.name else "unnamed"
            tunnel = fugue.tunnel.Tunnel(tunnel_name, interface, peers)
            print(tunnel)
            tunnel.persist()

        elif args.action == 'rm':
            fugue.tunnel.Tunnel.rm(args.name)

    elif args.command == 'interface':
        if args.action == 'new':
            interface = fugue.interface.Interface(args.private_key,
                                                  ipaddress.IPv4Network(args.address),
                                                  listen_port=args.listen_port,
                                                  name=args.name,
                                                  dns=[ipaddress.IPv4Address(dns) for dns in args.dns],
                                                  table=args.table,
                                                  mtu=args.mtu,
                                                  pre_ups=args.pre_ups,
                                                  post_ups=args.post_ups,
                                                  pre_downs=args.pre_downs,
                                                  post_downs=args.post_downs)
            print(interface)
            interface.persist()

        if args.action == 'show':
            interfaces = fugue.interface.Interface.get_interfaces()
            print(interfaces)

        if args.action == 'rm':
            fugue.interface.Interface.rm(args.name)

    elif args.command == 'peer':
        if args.action == 'new':
            peer = fugue.peer.Peer(args.public_key,
                                   [ipaddress.IPv4Network(aip) for aip in args.allowed_ips],
                                   name=args.name,
                                   endpoint=args.endpoint,
                                   persistent_keepalive=args.persistent_keepalive)
            print(peer)
            peer.persist()

        elif args.action == "show":
            peers = fugue.peer.Peer.get_peers()
            print(peers)

        elif args.action == "rm":
            fugue.peer.Peer.rm(args.name)


if __name__ == "__main__":
    main()
