import os
import os.path
import ipaddress
import pathlib
import fugue.utils
import fugue.subprocess.wg
from dataclasses import dataclass, KW_ONLY
from typing import List, Dict, Any, TypedDict, Optional


class InterfaceDict(TypedDict):
    private_key: str
    address: ipaddress.IPv4Network
    listen_port: Optional[int]
    name: Optional[str]
    dns: Optional[List[ipaddress.IPv4Address]]
    table: Optional[str]
    mtu: Optional[int]
    pre_ups: Optional[List[str]]
    post_ups: Optional[List[str]]
    pre_downs: Optional[List[str]]
    post_downs: Optional[List[str]]


@dataclass
class Interface:
    """
    A tunnel's interface.

    For configuration reference see: https://github.com/pirate/wireguard-docs#Interface
    """
    private_key: str
    address: ipaddress.IPv4Network
    _: KW_ONLY
    listen_port: Optional[int] = None
    name: Optional[str] = None
    dns: Optional[List[ipaddress.IPv4Address]] = None
    table: Optional[str] = None
    mtu: Optional[int] = None
    pre_ups: Optional[List[str]] = None
    post_ups: Optional[List[str]] = None
    pre_downs: Optional[List[str]] = None
    post_downs: Optional[List[str]] = None

    @staticmethod
    def from_conf(conf_fp: pathlib.Path) -> 'Interface':
        interfaces: List[fugue.interface.Interface] = []
        interfaces, _ = fugue.utils.parse_conf(conf_fp)

        # checks
        errs: List[str] = []
        if len(interfaces) == 0:
            errs.append(f"error: '{conf_fp}' doesn't define any Interface")

        if errs:
            raise RuntimeError(errs)

        return interfaces[0]

    @staticmethod
    def from_dict(_dict: InterfaceDict) -> 'Interface':  # https://stackoverflow.com/q/44640479
        return Interface(_dict["private_key"],
                         _dict["address"],
                         listen_port=_dict.get("listen_port"),
                         name=_dict.get("name"),
                         dns=_dict.get("dns"),
                         table=_dict.get("table"),
                         mtu=_dict.get("mtu"),
                         pre_ups=_dict.get("pre_ups"),
                         post_ups=_dict.get("post_ups"),
                         pre_downs=_dict.get("pre_downs"),
                         post_downs=_dict.get("post_downs"))

    @staticmethod
    def parse_section_line(_dict: Dict[str, Any], line: str) -> Dict[str, Any]:
        if line.startswith("# Name"):
            _dict["name"] = fugue.utils.ini_after_equal(line)

        elif line.startswith("Address"):
            _dict["address"] = ipaddress.IPv4Network(fugue.utils.ini_after_equal(line))

        elif line.startswith("ListenPort"):
            _dict["listen_port"] = int(fugue.utils.ini_after_equal(line))

        elif line.startswith("PrivateKey"):
            _dict["private_key"] = fugue.utils.ini_after_equal(line)

        elif line.startswith("DNS"):
            _dict["dns"] = [ipaddress.IPv4Address(dns.strip()) for dns in fugue.utils.ini_after_equal(line).split(',')]

        elif line.startswith("Table"):
            _dict["table"] = fugue.utils.ini_after_equal(line)

        elif line.startswith("MTU"):
            _dict["mtu"] = int(fugue.utils.ini_after_equal(line))

        elif line.startswith("PreUp"):
            _dict.setdefault("pre_ups", []).append(fugue.utils.ini_after_equal(line))

        elif line.startswith("PostUp"):
            _dict.setdefault("post_ups", []).append(fugue.utils.ini_after_equal(line))

        elif line.startswith("PreDown"):
            _dict.setdefault("pre_downs", []).append(fugue.utils.ini_after_equal(line))

        elif line.startswith("PostDown"):
            _dict.setdefault("post_downs", []).append(fugue.utils.ini_after_equal(line))

        return _dict

    def as_ini(self, print_header=True) -> str:
        ini_str: str
        if print_header:
            ini_str = (
                '# Generated by fugue\n'
                '[Interface]\n'
            )

        else:
            ini_str = '[Interface]\n'

        if self.name:
            ini_str += f'# Name = {self.name}\n'

        if self.address:
            ini_str += f'Address = {self.address}\n'

        if self.listen_port:
            ini_str += f'ListenPort = {self.listen_port}\n'

        if self.private_key:
            ini_str += f'PrivateKey = {self.private_key}\n'

        if self.dns:
            ini_str += f"DNS = {', '.join([str(d) for d in self.dns])}\n"

        if self.table:
            ini_str += f'Table = {self.table}\n'

        if self.mtu:
            ini_str += f'MTU = {self.mtu}\n'

        if self.pre_ups:
            for pre_up in self.pre_ups:
                ini_str += f'PreUp = {pre_up}\n'

        if self.post_ups:
            for post_up in self.post_ups:
                ini_str += f'PostUp = {post_up}\n'

        if self.pre_downs:
            for pre_down in self.pre_downs:
                ini_str += f'PreDown = {pre_down}\n'

        if self.post_downs:
            for post_down in self.post_downs:
                ini_str += f'PostDown = {post_down}\n'

        return ini_str

    def full_path(self) -> pathlib.Path:
        # TODO: how do we name the file if there is no name? let's do <public_key>.conf for now
        _basename: str = (
            f'{self.name}.conf'
            if self.name
            else f'{fugue.subprocess.wg.gen_public_key(self.private_key)}.conf'
        )
        return pathlib.Path(fugue.config.FUGUE_INTERFACES_ROOT, _basename)

    def persist(self) -> None:
        fugue.utils.write_protected(self.full_path(), self.as_ini())

    @classmethod
    def get_interfaces(cls) -> List['Interface']:
        interfaces: List['Interface'] = []
        conf_fps: List[pathlib.Path] = [pathlib.Path(fugue.config.FUGUE_INTERFACES_ROOT, child)
                                        for child in os.listdir(fugue.config.FUGUE_INTERFACES_ROOT)
                                        if pathlib.Path(fugue.config.FUGUE_INTERFACES_ROOT, child)
                                        and child.endswith('.conf')]

        for conf_fp in conf_fps:
            # TODO: we should try/catch here and report on malformed conf files
            interfaces.append(cls.from_conf(conf_fp))

        return interfaces

    @staticmethod
    def rm(name: str) -> None:
        conf_fp: pathlib.Path = pathlib.Path(fugue.config.FUGUE_INTERFACES_ROOT, f'{name}.conf')

        if not os.path.isfile(conf_fp):
            raise RuntimeError(f'error: interface with name: {name} not found at expected: {conf_fp}')

        os.remove(conf_fp)

    @classmethod
    def from_name(cls, name: str) -> 'Interface':
        interfaces: List['Interface'] = cls.get_interfaces()

        matching_interfaces: List['Interface'] = [p for p in interfaces if p.name == name]
        if len(matching_interfaces) == 0:
            raise RuntimeError('error: interface with name: {name} not found.')

        elif len(matching_interfaces) > 1:
            raise RuntimeError('error: found multiple interfaces with name: {name}: {matching_interfaces}')

        return matching_interfaces[0]
