import os
import ipaddress
import urllib.parse
import pathlib
import fugue.types
import fugue.utils
from dataclasses import dataclass, KW_ONLY
from typing import List, Optional, Dict, Any, TypedDict


class PeerDict(TypedDict):
    public_key: str
    allowed_ips: List[ipaddress.IPv4Network]
    name: Optional[str]
    endpoint: Optional[fugue.types.URL]
    persistent_keepalive: Optional[int]


@dataclass
class Peer:
    """
    Peer here relates to a remote peer's configuration, instead of a computer acting as a peer in a network.

    For configuration reference see: https://github.com/pirate/wireguard-docs#Peer
    """
    public_key: str
    allowed_ips: List[ipaddress.IPv4Network]
    _: KW_ONLY
    name: Optional[str] = None
    endpoint: Optional[fugue.types.URL] = None
    persistent_keepalive: Optional[int] = None

    @staticmethod
    def from_conf(conf_fp: pathlib.Path) -> 'Peer':
        peers: List[fugue.peer.Peer] = []
        _, peers = fugue.utils.parse_conf(conf_fp)

        # checks
        errs: List[str] = []
        if len(peers) == 0:
            errs.append(f"error: '{conf_fp}' doesn't define any Peer")

        if errs:
            raise RuntimeError(errs)

        return peers[0]

    @staticmethod
    def from_dict(_dict: PeerDict):
        return Peer(_dict["public_key"],
                    _dict["allowed_ips"],
                    name=_dict.get("name"),
                    endpoint=_dict.get("endpoint"),
                    persistent_keepalive=_dict.get("persistent_keepalive"))

    @staticmethod
    def parse_section_line(_dict: Dict[str, Any], line: str) -> Dict[str, Any]:
        if line.startswith("# Name"):
            _dict["name"] = fugue.utils.ini_after_equal(line)

        elif line.startswith("PublicKey"):
            _dict["public_key"] = fugue.utils.ini_after_equal(line)

        elif line.startswith("AllowedIPs"):
            _dict["allowed_ips"] = [ipaddress.IPv4Network(ip.strip())
                                    for ip in fugue.utils.ini_after_equal(line).split(',')]

        elif line.startswith("Endpoint"):
            _dict["endpoint"] = urllib.parse.urlparse(fugue.utils.ini_after_equal(line))

        elif line.startswith("PersistentKeepalive"):
            _dict["persistent_keepalive"] = int(fugue.utils.ini_after_equal(line))

        return _dict

    def as_ini(self, print_header=True) -> str:
        ini_str: str
        if print_header:
            ini_str = (
                '# Generated by fugue\n'
                '[Peer]\n'
            )

        else:
            ini_str = '[Peer]\n'

        if self.name:
            ini_str += f'# Name = {self.name}\n'

        ini_str += (
            f'PublicKey = {self.public_key}\n'
            f"AllowedIPs = {','.join([str(aip) for aip in self.allowed_ips])}\n"
        )

        if self.endpoint:
            ini_str += f'Endpoint = {self.endpoint.geturl()}\n'

        if self.persistent_keepalive:
            ini_str += f'PersistentKeepalive = {self.persistent_keepalive}\n'

        return ini_str

    def full_path(self) -> pathlib.Path:
        # TODO: how do we name the file if there is no name?
        _basename: str = f'{self.name}.conf'
        return pathlib.Path(fugue.config.FUGUE_PEERS_ROOT, _basename)

    def persist(self) -> None:
        fugue.utils.write_protected(self.full_path(), self.as_ini())

    @classmethod
    def get_peers(cls) -> List['Peer']:
        peers: List['Peer'] = []
        conf_fps: List[pathlib.Path] = [pathlib.Path(fugue.config.FUGUE_PEERS_ROOT, child)
                                        for child in os.listdir(fugue.config.FUGUE_PEERS_ROOT)
                                        if pathlib.Path(fugue.config.FUGUE_PEERS_ROOT, child)
                                        and child.endswith('.conf')]

        for conf_fp in conf_fps:
            # TODO: we should try/catch here and report on malformed conf files
            peers.append(cls.from_conf(conf_fp))

        return peers

    @staticmethod
    def rm(name: str) -> None:
        conf_fp: pathlib.Path = pathlib.Path(fugue.config.FUGUE_PEERS_ROOT, f'{name}.conf')

        if not os.path.isfile(conf_fp):
            raise RuntimeError(f'error: peer with name: {name} not found at expected: {conf_fp}')

        os.remove(conf_fp)

    @classmethod
    def from_name(cls, name: str) -> 'Peer':
        peers: List['Peer'] = cls.get_peers()

        matching_peers: List['Peer'] = [p for p in peers if p.name == name]
        if len(matching_peers) == 0:
            raise RuntimeError('error: peer with name: {name} not found')

        elif len(matching_peers) > 1:
            raise RuntimeError('error: found multiple peers with name: {name}: {matching_peers}')

        return matching_peers[0]
