import os
import ipaddress
import random
import pathlib
from dataclasses import dataclass
from typing import List

from fugue import config
from fugue.lib import Interface, Peer, Tunnel, utils, types
from fugue.lib.subprocess import wg


"""
What's a Hub and Spoke topology[1]
Configure Wireguard for Hub and Spoke topology[2]

[1] https://www.procustodibus.com/blog/2020/10/wireguard-topologies/#hub-and-spoke
[2] https://www.procustodibus.com/blog/2020/11/wireguard-hub-and-spoke-config/
"""


@dataclass
class Hub:
    name: str
    tunnel: Tunnel
    as_peer: Peer
    dir_path: pathlib.Path
    spokes_dir_path: pathlib.Path

    @staticmethod
    def gen(name: str,
            listen_port: int,
            endpoint: types.URL) -> 'Hub':
        # TODO: an interactive mode?
        interface: Interface

        private_key: str = wg.gen_private_key()
        gen_network_str: str = f'10.{random.randint(0,255)}.{random.randint(0,255)}.0/24'
        gen_network: ipaddress.IPv4Network = ipaddress.IPv4Network(gen_network_str)

        # hub interface
        # NOTE: Pro Custodibus hints at an address with a /32 mask, when the unofficial docs say that for "bounce
        # servers", we can set an address with the relevant VPN CIDR block mask.
        # See: https://www.procustodibus.com/blog/2020/11/wireguard-hub-and-spoke-config/#configure-routing-on-host-c
        # And: https://github.com/pirate/wireguard-docs#address
        interface = Interface(
            private_key,
            ipaddress.IPv4Network(f'{gen_network.network_address}/24'),
            listen_port=listen_port,
            name=name,
            dns=[gen_network.network_address],
            table=None,
            mtu=None,
            pre_ups=['sysctl -w net.ipv4.ip_forward=1'],
            post_ups=None,
            pre_downs=None,
            post_downs=None)

        # hub as remote peer
        # NOTE: if the Hub is behind a NAT, it looks like it'd be relevant to send a persistent keepalive.
        # I'm not one hunder percent on this though.
        # See: https://www.procustodibus.com/blog/2020/11/wireguard-hub-and-spoke-config/#configure-routing-on-host-c
        # (on Pro Custodibus, the diagram clearly shows the hub as being behind a NAT)
        # See on unofficial docs: https://github.com/pirate/wireguard-docs#PersistentKeepalive
        as_peer: Peer = Peer(
            wg.gen_public_key(private_key),
            [gen_network],
            name=name,
            endpoint=endpoint,
            persistent_keepalive=25
        )

        tunnel: Tunnel = Tunnel(interface, [])
        dir_path: pathlib.Path = pathlib.Path(config.FUGUE_HUBS_ROOT, f'{name}.d')
        spokes_dir_path: pathlib.Path = pathlib.Path(dir_path, 'spokes')

        return Hub(name, tunnel, as_peer, dir_path, spokes_dir_path)

    def rm(self) -> None:
        # removes hub
        # remove tunnel's interface
        Interface.rm(self.tunnel.interface.name)
        # remove tunnel
        Tunnel.rm(self.tunnel.get_name())
        # remove as_peer
        Peer.rm(self.as_peer.name)
        # remove directory
        utils.rm_tree(self.dir_path)

        # TODO: Should we remove the hubs directory if it's become empty?
        utils.rm_dir_if_empty(config.FUGUE_HUBS_ROOT)

    def persist(self) -> None:
        # persist tunnel, interface and peer objects
        self.tunnel.interface.persist()
        for peer in self.tunnel.peers:
            peer.persist()

        self.tunnel.persist()
        self.as_peer.persist()

        # make hub config directory
        utils.makedir_protected(self.dir_path)
        utils.makedir_protected(self.spokes_dir_path)

        # symlink tunnel and as peer config files
        fp_tunnel: pathlib.Path = pathlib.Path(self.dir_path, 'tunnel.conf')
        fp_as_peer: pathlib.Path = pathlib.Path(self.dir_path, 'as_peer.conf')

        os.symlink(self.tunnel.full_path(), fp_tunnel)
        os.symlink(self.as_peer.full_path(), fp_as_peer)

    @staticmethod
    def get_hubs() -> List['Hub']:
        hubs: List['Hub'] = []

        hubs_dirs: List[pathlib.Path] = [
            pathlib.Path(config.FUGUE_HUBS_ROOT, child)
            for child in os.listdir(config.FUGUE_HUBS_ROOT)
            if (
                pathlib.Path(config.FUGUE_HUBS_ROOT, child)
                and child.endswith('.d')
                and os.path.isdir(pathlib.Path(config.FUGUE_HUBS_ROOT, child)))
        ]

        name: str
        tunnel_conf_fp: pathlib.Path
        as_peer_conf_fp: pathlib.Path
        tunnel: Tunnel
        as_peer: Peer
        spokes_dir_path: pathlib.Path

        for hub_dir in hubs_dirs:
            tunnel_conf_fp = pathlib.Path(hub_dir, 'tunnel.conf')
            as_peer_conf_fp = pathlib.Path(hub_dir, 'as_peer.conf')

            tunnel = Tunnel.from_conf(tunnel_conf_fp)
            name = tunnel.get_name()
            as_peer = Peer.from_conf(as_peer_conf_fp)

            spokes_dir_path = pathlib.Path(hub_dir, 'spokes')

            hubs.append(Hub(name, tunnel, as_peer, hub_dir, spokes_dir_path))

        return hubs

    @staticmethod
    def from_name(name: str) -> 'Hub':
        hubs: List['Hub'] = Hub.get_hubs()

        matching_hubs: List['Hub'] = [h for h in hubs if h.name == name]
        if len(matching_hubs) == 0:
            raise RuntimeError('error: interface with name: {name} not found.')

        elif len(matching_hubs) > 1:
            raise RuntimeError('error: found multiple hubs with name: {name}: {matching_hubs}')

        return matching_hubs[0]

    def start(self):
        self.tunnel.start()

    def stop(self):
        self.tunnel.stop()

    def enable(self):
        self.tunnel.enable()

    def disable(self):
        self.tunnel.disable()
