import os
import pathlib
from dataclasses import dataclass
from typing import List
import fugue.interface
import fugue.peer
import fugue.utils
import fugue.subprocess.systemctl


@dataclass
class Tunnel:
    """
    An interface and list of peers.
    """
    name: str
    interface: fugue.interface.Interface
    peers: List[fugue.peer.Peer]

    @staticmethod
    def from_conf(conf_fp: pathlib.Path):
        interfaces: List[fugue.interface.Interface] = []
        peers: List[fugue.peer.Peer] = []
        interfaces, peers = fugue.utils.parse_conf(conf_fp)

        # checks
        errs: List[str] = []
        if len(interfaces) > 1:
            errs.append(f"error: Tunnel at: '{conf_fp}' can't define more than one Interface")

        if len(interfaces) == 0:
            errs.append(f"error: Tunnel at: '{conf_fp}' doesn't define any Interface")

        if errs:
            raise RuntimeError(errs)

        # grabbing interface
        interface: fugue.interface.Interface = interfaces[0]

        # NOTE: is a tunnel's name really its interface's name?
        tunnel_name: str = interface.name if interface.name else "unnamed"
        return Tunnel(tunnel_name, interface, peers)

    def full_path(self) -> pathlib.Path:
        _basename: str = f'{self.name}.conf'
        return pathlib.Path(fugue.config.FUGUE_TUNNELS_ROOT, _basename)

    def as_ini(self) -> str:
        ini_str: str = self.interface.as_ini(print_header=False)

        for peer in self.peers:
            ini_str += '\n'
            ini_str += peer.as_ini(print_header=False)

        return ini_str

    def persist(self) -> None:
        fugue.utils.write_protected(self.full_path(), self.as_ini())

    def add_peer(self, peer: fugue.peer.Peer) -> None:
        self.peers.append(peer)
        # TODO: is materialized? If so, concatenate to file? Or rewrite the file?
        # TODO: is running?

    @staticmethod
    def rm(name: str) -> None:
        conf_fp: pathlib.Path = pathlib.Path(fugue.config.FUGUE_TUNNELS_ROOT, f'{name}.conf')

        if not os.path.isfile(conf_fp):
            raise RuntimeError(f'error: tunnel with name: {name} not found at expected: {conf_fp}')

        os.remove(conf_fp)

    @staticmethod
    def get_tunnels() -> List['Tunnel']:
        tunnels: List['Tunnel'] = []
        conf_fps: List[pathlib.Path] = [pathlib.Path(fugue.config.FUGUE_TUNNELS_ROOT, child)
                                        for child in os.listdir(fugue.config.FUGUE_TUNNELS_ROOT)
                                        if pathlib.Path(fugue.config.FUGUE_TUNNELS_ROOT, child)
                                        and child.endswith('.conf')]

        for conf_fp in conf_fps:
            # TODO: we should try/catch here and report on malformed conf files
            tunnels.append(Tunnel.from_conf(conf_fp))

        return tunnels

    @staticmethod
    def from_name(name: str) -> 'Tunnel':
        tunnels: List['Tunnel'] = Tunnel.get_tunnels()
        return [tunnel for tunnel in tunnels if tunnel.name == name].pop()

    def is_promoted(self) -> bool:
        # NOTE: A tunnel is considered promoted if there's a symlink to it under /etc/wireguard/, e.g. it can be managed
        # with wg-quick.
        return os.path.islink(pathlib.Path(fugue.config.WG_ROOT, f'{self.name}.conf'))

    def promote(self) -> None:
        if not self.is_promoted():
            os.symlink(self.full_path(), pathlib.Path(fugue.config.WG_ROOT, f'{self.name}.conf'))

    def demote(self) -> None:
        if self.is_promoted():
            os.unlink(pathlib.Path(fugue.config.WG_ROOT, f'{self.name}.conf'))

    def is_running(self) -> bool:
        status: str = fugue.subprocess.wg.status()
        interface_lines: List[str] = [line.strip() for line in status.split('\n')]

        # NOTE: we don't wanna use systemctl to check if a tunnel is up anymore, see todo notes
        # status: str = fugue.subprocess.systemctl.status(f'wg-quick@{self.name}.service')
        # status_lines: List[str] = [line.strip() for line in status.split('\n')]
        # active_line: str = [line for line in status_lines if line.startswith('Active:')].pop()

        return f"interface: {self.name}" in interface_lines

    def start(self):
        if self.is_running():
            return

        if not self.is_promoted():
            self.promote()

        fugue.subprocess.wg.quick("up", f"{self.name}")
        # NOTE: we don't wanna use systemctl to start/stop tunnels, but wg-quick
        # fugue.subprocess.systemctl.start(f"wg-quick@{self.name}.service")

    def stop(self):
        if self.is_running():
            fugue.subprocess.wg.quick("down", f"{self.name}")
            # NOTE: we don't wanna use systemctl to start/stop tunnels, but wg-quick
            # fugue.subprocess.systemctl.stop(f"wg-quick@{self.name}.service")

    def enable(self):
        if not self.is_promoted():
            self.promote()
        fugue.subprocess.systemctl.enable(f"wg-quick@{self.name}.service", now=True)

    def disable(self):
        if not self.is_promoted():
            self.promote()
        fugue.subprocess.systemctl.disable(f"wg-quick@{self.name}.service", now=True)
