import argparse
import ipaddress
import urllib.parse
import textwrap
from typing import List, cast

import fugue.config
from fugue.lib import utils, Interface, Peer, Tunnel, Hub, Spoke

from fugue.cli import show as cli_show
from fugue.cli import config as cli_config
from fugue.cli import interface as cli_interface
from fugue.cli import peer as cli_peer
from fugue.cli import tunnel as cli_tunnel
from fugue.cli import hub as cli_hub
from fugue.cli import spoke as cli_spoke
from fugue.cli import explain as cli_explain


# TODO: what license to use? I don't want people to use this to make money.
# They should also have to republish modifications under the same license. I
# guess I'd also want modifications to be open-source.
#
# I think that's helpful?
# https://opensource.stackexchange.com/questions/4875/open-source-license-to-prevent-commercial-use  # noqa: E501
#
# I think this didn't really have what I was looking for: https://man.sr.ht/license.md.

# TODO: can we run checks to ensure that a site's port is properly forwarded through the router?
# TODO: can we run checks to make sure a site's endpoint is reachable?
# TODO: can we run checks to make sure a site's has got dynamic DNS properly setup?
# TODO: should we check that the site is properly set as the LAN's default DNS, if possible?
# TODO: is any of the above NOT in the scope of fugue?


def parse_args(init_interfaces: List[Interface],
               init_peers: List[Peer],
               init_tunnels: List[Tunnel],
               init_hubs: List[Hub]) -> argparse.Namespace:
    parser = argparse.ArgumentParser()

    subparsers = parser.add_subparsers(help="Fugue command", dest="command")

    # the subparser had subparsers
    cli_explain.set_subparser(cast(argparse.Namespace, subparsers))
    cli_show.set_subparser(cast(argparse.Namespace, subparsers))
    cli_config.set_subparser(cast(argparse.Namespace, subparsers))
    cli_interface.set_subparser(cast(argparse.Namespace, subparsers), init_interfaces)
    cli_peer.set_subparser(cast(argparse.Namespace, subparsers), init_peers)
    cli_tunnel.set_subparser(cast(argparse.Namespace, subparsers), init_tunnels, init_interfaces, init_peers)
    cli_hub.set_subparser(cast(argparse.Namespace, subparsers))
    cli_spoke.set_subparser(cast(argparse.Namespace, subparsers), init_hubs)

    return parser.parse_args()


def cli() -> None:
    utils.check_privileges()

    utils.makedir_protected(fugue.config.FUGUE_ROOT)
    utils.makedir_protected(fugue.config.FUGUE_INTERFACES_ROOT)
    utils.makedir_protected(fugue.config.FUGUE_PEERS_ROOT)
    utils.makedir_protected(fugue.config.FUGUE_TUNNELS_ROOT)
    utils.makedir_protected(fugue.config.FUGUE_HUBS_ROOT)

    init_interfaces: List[Interface] = Interface.get_interfaces()
    init_peers: List[Peer] = Peer.get_peers()
    init_tunnels: List[Tunnel] = Tunnel.get_tunnels()
    init_hubs: List[Hub] = Hub.get_hubs()

    args = parse_args(init_interfaces, init_peers, init_tunnels, init_hubs)
    cli_interface.check_args(args)

    # init vars
    interfaces: List[Interface]
    peers: List[Peer]
    tunnels: List[Tunnel]
    hubs: List[Hub]
    spokes: List[Spoke]

    interface: Interface
    peer: Peer
    tunnel: Tunnel
    hub: Hub
    spoke: Spoke

    # explain
    # ==========================================================================
    if args.command == "explain":
        # we'll just cover the basics for now
        interfaces = Interface.get_interfaces()
        hubs = Hub.get_hubs()
        spokes = Spoke.get_spokes()
        peers = Peer.get_peers()

        # 1. peer acts as a hub
        message: str = ""
        if len(hubs) == 1:
            hub = hubs[0]
            hub_address = hub.tunnel.interface.address
            hub_listen_port = hub.tunnel.interface.listen_port
            hub_endpoint = hub.as_peer.endpoint.path

            message += textwrap.fill(
                f"This computer acts as a Hub with name: '{hub.name}' for IP "
                f"addresses belonging to the subnet: {hub_address}. It is "
                f"listening on the port: {hub_listen_port}, and it "
                f"currently serves {len(hub.tunnel.peers)} Spoke(s).",
                width=80
            )

            message += "\n\n"

            message += textwrap.fill(
                f"It should be accessible at the endpoint: {hub_endpoint}. "
                "If it is running behind a NAT device, remember to set the "
                "proper port-forwarding in place; e.g. from any port on the "
                "router, e.g. 45678, to this computer's private IP address "
                f"on port: {hub_listen_port}. Running 'ip addr' should tell "
                "what the private IP is.",
                width=80
            )

            message += "\n\n"

            if hub.tunnel.is_running():
                message += textwrap.fill(
                    "It is currently running and accepting connections. It "
                    "can be stopped with the CLI by typing:\n\n"
                    f"fugue stop -n {hub.name}",
                    width=80
                )

            else:
                message += textwrap.fill(
                    "It is currently not running. It can be started with the "
                    "CLI by typing:",
                    width=80
                )
                message += "\n\n"
                message += textwrap.fill(
                    f"fugue start -n {hub.name}",
                    width=80
                )

        print(message)

    # show
    # ==========================================================================
    if args.command == "show":
        # show exhaustive list of resources
        interfaces = Interface.get_interfaces()
        hubs = Hub.get_hubs()
        spokes = Spoke.get_spokes()
        peers = Peer.get_peers()
        tunnels = Tunnel.get_tunnels()

        fugue.config.show()
        print()
        print(f"interfaces ({len(interfaces)}): {interfaces}")
        print()
        print(f"peers ({len(peers)}): {peers}")
        print()
        print(f"tunnels ({len(tunnels)}): {tunnels}")
        print()
        print(f"hubs ({len(hubs)}): {hubs}")
        print()
        print(f"spokes ({len(spokes)}): {spokes}")

    # config
    # ==========================================================================
    elif args.command == "config":
        if args.action == "show":
            fugue.config.show()

    # Interface
    # ==========================================================================
    elif args.command == "interface":
        if args.action == "new":
            interface = Interface(args.private_key,
                                  ipaddress.IPv4Network(args.address),
                                  listen_port=args.listen_port,
                                  name=args.name,
                                  dns=[ipaddress.IPv4Address(dns) for dns in args.dns],
                                  table=args.table,
                                  mtu=args.mtu,
                                  pre_ups=args.pre_ups,
                                  post_ups=args.post_ups,
                                  pre_downs=args.pre_downs,
                                  post_downs=args.post_downs)
            print(interface)
            interface.persist()

        elif args.action == "show":
            interfaces = Interface.get_interfaces()
            print(interfaces)

        elif args.action == "rm":
            Interface.rm(args.name)

        elif args.action == "set":
            # TODO: do we need to check whether that interface is used by any tunnel?
            interface = Interface.from_name(args.name)
            if args.private_key:
                interface.private_key = args.private_key

            if args.address:
                interface.address = ipaddress.IPv4Network(args.address)

            if args.listen_port:
                interface.listen_port = args.listen_port

            if args.name:
                # TODO: if we change the name of the interface, should the filename (or any symlink) be cleaned up?
                # If we wanted **not to**, then filename should not depend on the # Name attribute
                # Which prompts the question, are any "Names" unique in fugue? How about hub's names?
                # The only unique thing about a tunnel, interface or peer, is the combination of its attributes
                interface.name = args.name

            if args.dns:
                interface.dns = [ipaddress.IPv4Address(dns) for dns in args.dns]

            if args.private_key:
                interface.private_key = args.private_key

            if args.private_key:
                interface.private_key = args.private_key

            if args.private_key:
                interface.private_key = args.private_key

    # Peer
    # ==========================================================================
    elif args.command == "peer":
        if args.action == "new":
            peer = Peer(args.public_key,
                        [ipaddress.IPv4Network(aip) for aip in args.allowed_ips],
                        name=args.name,
                        endpoint=args.endpoint,
                        persistent_keepalive=args.persistent_keepalive)
            print(peer)
            peer.persist()

        elif args.action == "show":
            peers = Peer.get_peers()
            print(peers)

        elif args.action == "rm":
            Peer.rm(args.name)

    # Tunnel
    # ==========================================================================
    elif args.command == "tunnel":
        if args.action == "show":
            tunnels = Tunnel.get_tunnels()

            print(f"tunnels: {tunnels}")

        elif args.action == "new":
            interface = Interface.from_name(args.interface_name)
            peers = []
            for peer_name in args.peer_names:
                peers.append(Peer.from_name(peer_name))

            # TODO: how to properly handle names when those do not exist?
            tunnel = Tunnel(interface, peers)
            print(tunnel)
            tunnel.persist()

        elif args.action == "rm":
            Tunnel.rm(args.name)

        elif args.action == "start":
            tunnel = Tunnel.from_name(args.name)
            tunnel.start()

        elif args.action == "stop":
            tunnel = Tunnel.from_name(args.name)
            tunnel.stop()

    # Hub
    # ==========================================================================
    elif args.command == "hub":
        if args.action == "gen":
            hub = Hub.gen(args.name, args.listen_port, urllib.parse.urlparse(args.endpoint))
            print(hub)
            hub.persist()

        elif args.action == "rm":
            hub = Hub.from_name(args.name)
            hub.rm()

        elif args.action == "show":
            hubs = Hub.get_hubs()
            print(f"hubs: {hubs}")

        elif args.action == "start":
            hub = Hub.from_name(args.name)
            hub.tunnel.start()

        elif args.action == "stop":
            hub = Hub.from_name(args.name)
            hub.tunnel.stop()

    # Spoke
    # ==========================================================================
    elif args.command == "spoke":
        if args.action == "gen":
            hub = Hub.from_name(args.hub_name)
            spoke = Spoke.gen(args.name, hub)
            print(spoke)
            spoke.persist()
            spoke.hub.tunnel.persist()

        elif args.action == "show":
            if args.hub_name:
                hub = Hub.from_name(args.hub_name)
                spokes = Spoke.get_spokes_for_hub(hub)

            else:
                spokes = Spoke.get_spokes()

            print(spokes)


if __name__ == "__main__":
    cli()
