import os
import ipaddress
import urllib.parse
import pathlib
from dataclasses import dataclass, KW_ONLY
from typing import List, Optional, Dict, Any, cast

from fugue import config
from fugue.lib import types, utils


@dataclass
class Peer:
    """
    Peer here relates to a remote peer's configuration, instead of a computer acting as a peer in a network.

    For configuration reference see: https://github.com/pirate/wireguard-docs#Peer
    """
    public_key: str
    allowed_ips: List[ipaddress.IPv4Network]
    _: KW_ONLY
    name: Optional[str] = None
    endpoint: Optional[types.URL] = None
    persistent_keepalive: Optional[int] = None

    @staticmethod
    def from_dict(_dict: types.PeerDict):
        return Peer(_dict["public_key"],
                    _dict["allowed_ips"],
                    name=_dict.get("name"),
                    endpoint=_dict.get("endpoint"),
                    persistent_keepalive=_dict.get("persistent_keepalive"))

    @staticmethod
    def parse_conf_line(acc: Dict[str, Any], line: str) -> Dict[str, Any]:
        # update accumulator with [Peer] section line
        if line.startswith("# Name"):
            acc["name"] = utils.ini_after_equal(line)

        elif line.startswith("PublicKey"):
            acc["public_key"] = utils.ini_after_equal(line)

        elif line.startswith("AllowedIPs"):
            acc["allowed_ips"] = [ipaddress.IPv4Network(ip.strip())
                                  for ip in utils.ini_after_equal(line).split(',')]

        elif line.startswith("Endpoint"):
            acc["endpoint"] = urllib.parse.urlparse(utils.ini_after_equal(line))

        elif line.startswith("PersistentKeepalive"):
            acc["persistent_keepalive"] = int(utils.ini_after_equal(line))

        return acc

    @staticmethod
    def from_conf(conf_fp: pathlib.Path) -> 'Peer':
        with open(conf_fp, 'r') as fd:
            conf_lines = fd.readlines()

        # init vars
        in_peer_section: bool = False

        # accumulator
        acc: Dict[str, Any] = {}

        # cleaning
        conf_lines_cleaned = [line.replace('\n', '').strip()  # newline chars & strip
                              for line in conf_lines]
        conf_lines_cleaned = [line
                              for line in conf_lines_cleaned
                              # empty lines & comments, except for Name line
                              if line and (not line.startswith('#') or line.startswith('# Name'))]

        for line in conf_lines_cleaned:
            # init
            if not in_peer_section:
                if line.startswith("[Peer]"):
                    in_peer_section = True

                else:
                    raise RuntimeError(f"error: parsing peer conf file: '{conf_fp}'")

            # parse a peer section
            elif in_peer_section:
                # error: entering unrecognised section
                if line.startswith('['):
                    raise RuntimeError(f"error: unrecognised section: '{line}' in interface file: '{conf_fp}'")

                else:
                    # update accumulator with [Peer] section line
                    acc = Peer.parse_conf_line(acc, line)

        return Peer.from_dict(cast(types.PeerDict, acc))

    def as_ini(self, print_header=True) -> str:
        ini_str: str
        if print_header:
            ini_str = (
                '# Generated by fugue\n'
                '[Peer]\n'
            )

        else:
            ini_str = '[Peer]\n'

        if self.name:
            ini_str += f'# Name = {self.name}\n'

        ini_str += (
            f'PublicKey = {self.public_key}\n'
            f"AllowedIPs = {','.join([str(aip) for aip in self.allowed_ips])}\n"
        )

        if self.endpoint:
            ini_str += f'Endpoint = {self.endpoint.geturl()}\n'

        if self.persistent_keepalive:
            ini_str += f'PersistentKeepalive = {self.persistent_keepalive}\n'

        return ini_str

    def full_path(self) -> pathlib.Path:
        # TODO: how do we name the file if there is no name?
        _basename: str = f'{self.name}.conf'
        return pathlib.Path(config.FUGUE_PEERS_ROOT, _basename)

    def persist(self) -> None:
        utils.write_protected(self.full_path(), self.as_ini())

    @staticmethod
    def get_peers() -> List['Peer']:
        peers: List['Peer'] = []
        conf_fps: List[pathlib.Path] = [pathlib.Path(config.FUGUE_PEERS_ROOT, child)
                                        for child in os.listdir(config.FUGUE_PEERS_ROOT)
                                        if pathlib.Path(config.FUGUE_PEERS_ROOT, child)
                                        and child.endswith('.conf')]

        for conf_fp in conf_fps:
            # TODO: we should try/catch here and report on malformed conf files
            peers.append(Peer.from_conf(conf_fp))

        return peers

    @staticmethod
    def rm(name: str) -> None:
        conf_fp: pathlib.Path = pathlib.Path(config.FUGUE_PEERS_ROOT, f'{name}.conf')

        if not os.path.isfile(conf_fp):
            raise RuntimeError(f'error: peer with name: {name} not found at expected: {conf_fp}')

        os.remove(conf_fp)

        # TODO: Should we remove the peers directory if it's become empty?
        utils.rm_dir_if_empty(config.FUGUE_PEERS_ROOT)

    @staticmethod
    def from_name(name: str) -> 'Peer':
        peers: List['Peer'] = Peer.get_peers()

        matching_peers: List['Peer'] = [p for p in peers if p.name == name]
        if len(matching_peers) == 0:
            raise RuntimeError('error: peer with name: {name} not found')

        elif len(matching_peers) > 1:
            raise RuntimeError('error: found multiple peers with name: {name}: {matching_peers}')

        return matching_peers[0]
