import os
import ipaddress
import pathlib
from dataclasses import dataclass
from typing import List

import qrcode
from qrcode.image.styledpil import StyledPilImage
from qrcode.image.styles.moduledrawers.pil import RoundedModuleDrawer
from qrcode.image.styles.colormasks import RadialGradiantColorMask

from fugue.lib import Interface, Peer, Tunnel, Hub, utils
from fugue.lib.subprocess import wg


@dataclass
class Spoke:
    name: str
    tunnel: Tunnel
    as_peer: Peer
    hub: Hub

    @staticmethod
    def gen(name: str, hub: Hub) -> 'Spoke':
        # new spoke address, and allowed_ip is max() + 1
        hub_peers_allowed_ips: List[List[ipaddress.IPv4Network]] = [
            peer.allowed_ips for peer in hub.tunnel.peers
        ]
        hub_peers_network_addresses: List[ipaddress.IPv4Address] = [
            allowed_ip.network_address
            for allowed_ips in hub_peers_allowed_ips
            for allowed_ip in allowed_ips
        ]

        address: ipaddress.IPv4Network
        if hub_peers_network_addresses:
            address = ipaddress.IPv4Network(f'{max(hub_peers_network_addresses) + 1}/32')
        else:
            address = ipaddress.IPv4Network(f'{hub.tunnel.interface.address.network_address + 1}/32')

        private_key: str = wg.gen_private_key()

        # TODO: what do we do for listen port? It should only be on the interface btw
        # It looks like we shouldn't set it: https://github.com/pirate/wireguard-docs#ListenPort
        interface: Interface = Interface(
            private_key,
            address,
            name=name,
            dns=[hub.tunnel.interface.address.network_address]
        )

        # NOTE: We can only assumes that peers will be road-warriors, hence will be connecting to the Hub from behind
        # NATs. As a result, we set a persistent keepalive to 25 seconds.
        as_peer: Peer = Peer(
            wg.gen_public_key(private_key),
            allowed_ips=[address],
            name=name,
            persistent_keepalive=25
        )

        tunnel: Tunnel = Tunnel(interface, [hub.as_peer])

        # add peer to hub's tunnel
        hub.tunnel.add_peer(as_peer)
        # TODO: if tunnel is up, should we do the update as in the legacy thing?

        return Spoke(name, tunnel, as_peer, hub)

    def persist(self) -> None:
        self.as_peer.persist()
        self.tunnel.persist()
        self.tunnel.interface.persist()
        # TODO: is the hub already persisted? Do we need a Hub.is_persisted method?
        # If it is, the spoke's tunnel's peer should already be persisted (as it's the `as_peer` peer of the Hub)
        # If the spoke gets added a peer (for some reason I'm not too sure of yet)
        # Something like that maybe?
        # for peer in self.tunnel.peers:
        #     if not peer.is_persisted:
        #         peer.persist()

        # make spoke config directory
        spoke_dir: pathlib.Path = pathlib.Path(self.hub.spokes_dir_path, f'{self.name}.d')
        utils.makedir_protected(spoke_dir)

        # symlink tunnel and as peer config files, make QR code for phones
        fp_tunnel: pathlib.Path = pathlib.Path(spoke_dir, 'tunnel.conf')
        fp_tunnel_png: pathlib.Path = pathlib.Path(spoke_dir, 'tunnel.conf.png')
        fp_as_peer: pathlib.Path = pathlib.Path(spoke_dir, 'as_peer.conf')

        os.symlink(self.tunnel.full_path(), fp_tunnel)
        os.symlink(self.as_peer.full_path(), fp_as_peer)

        qr = qrcode.QRCode(error_correction=qrcode.constants.ERROR_CORRECT_L)
        qr.add_data(self.tunnel.as_ini())
        qr.make(fit=True)

        # we add some style! https://github.com/lincolnloop/python-qrcode, I like RoundedModuleDrawer()
        # NOTE: adding style adds about 1 second latency to the peer creation process, but it's kinda pretty

        img = qr.make_image(back_color=(39, 119, 255),
                            fill_color=(248, 249, 251),
                            image_factory=StyledPilImage,
                            module_drawer=RoundedModuleDrawer(),
                            color_mask=RadialGradiantColorMask())
        img.save(fp_tunnel_png)

    @staticmethod
    def get_spokes_for_hub(hub: Hub) -> List['Spoke']:
        spokes: List['Spoke'] = []
        tunnel: Tunnel
        as_peer: Peer
        name: str

        spokes_dirs = [
            pathlib.Path(hub.spokes_dir_path, child)
            for child in os.listdir(hub.spokes_dir_path)
            if pathlib.Path(hub.spokes_dir_path)
            and child.endswith('.d')
            and os.path.isdir(pathlib.Path(hub.spokes_dir_path, child))
        ]

        for spoke_dir in spokes_dirs:
            name = spoke_dir.with_suffix('').name  # removing extension (e.g. '.d')
            tunnel = Tunnel.from_conf(pathlib.Path(spoke_dir, 'tunnel.conf'))
            as_peer = Peer.from_conf(pathlib.Path(spoke_dir, 'as_peer.conf'))
            spokes.append(Spoke(name, tunnel, as_peer, hub))

        return spokes

    @staticmethod
    def get_spokes() -> List['Spoke']:
        hubs: List[Hub] = Hub.get_hubs()
        spokes: List['Spoke'] = []

        for hub in hubs:
            spokes.extend(Spoke.get_spokes_for_hub(hub))

        return spokes
