import os
import os.path
import ipaddress
import pathlib
from dataclasses import dataclass, KW_ONLY
from typing import List, Dict, Any, Optional, cast

from fugue import config
from fugue.lib import utils, types
from fugue.lib.subprocess import wg


@dataclass
class Interface:
    """
    A tunnel's interface.

    For configuration reference see: https://github.com/pirate/wireguard-docs#Interface
    """
    private_key: str
    address: ipaddress.IPv4Network
    _: KW_ONLY
    listen_port: Optional[int] = None
    name: Optional[str] = None
    dns: Optional[List[ipaddress.IPv4Address]] = None
    table: Optional[str] = None
    mtu: Optional[int] = None
    pre_ups: Optional[List[str]] = None
    post_ups: Optional[List[str]] = None
    pre_downs: Optional[List[str]] = None
    post_downs: Optional[List[str]] = None

    @staticmethod
    def from_dict(_dict: types.InterfaceDict) -> 'Interface':  # https://stackoverflow.com/q/44640479
        return Interface(_dict["private_key"],
                         _dict["address"],
                         listen_port=_dict.get("listen_port"),
                         name=_dict.get("name"),
                         dns=_dict.get("dns"),
                         table=_dict.get("table"),
                         mtu=_dict.get("mtu"),
                         pre_ups=_dict.get("pre_ups"),
                         post_ups=_dict.get("post_ups"),
                         pre_downs=_dict.get("pre_downs"),
                         post_downs=_dict.get("post_downs"))

    @staticmethod
    def parse_conf_line(acc: Dict[str, Any], line: str) -> Dict[str, Any]:
        # update accumulator with [Interface] section line
        if line.startswith("# Name"):
            acc["name"] = utils.ini_after_equal(line)

        elif line.startswith("Address"):
            acc["address"] = ipaddress.IPv4Network(utils.ini_after_equal(line))

        elif line.startswith("ListenPort"):
            acc["listen_port"] = int(utils.ini_after_equal(line))

        elif line.startswith("PrivateKey"):
            acc["private_key"] = utils.ini_after_equal(line)

        elif line.startswith("DNS"):
            acc["dns"] = [ipaddress.IPv4Address(dns.strip()) for dns in utils.ini_after_equal(line).split(',')]

        elif line.startswith("Table"):
            acc["table"] = utils.ini_after_equal(line)

        elif line.startswith("MTU"):
            acc["mtu"] = int(utils.ini_after_equal(line))

        elif line.startswith("PreUp"):
            acc.setdefault("pre_ups", []).append(utils.ini_after_equal(line))

        elif line.startswith("PostUp"):
            acc.setdefault("post_ups", []).append(utils.ini_after_equal(line))

        elif line.startswith("PreDown"):
            acc.setdefault("pre_downs", []).append(utils.ini_after_equal(line))

        elif line.startswith("PostDown"):
            acc.setdefault("post_downs", []).append(utils.ini_after_equal(line))

        return acc

    @staticmethod
    def from_conf(conf_fp: pathlib.Path) -> 'Interface':
        with open(conf_fp, 'r') as fd:
            conf_lines = fd.readlines()

        # init vars
        in_interface_section: bool = False

        # accumulator
        acc: Dict[str, Any] = {}

        # cleaning
        conf_lines_cleaned = [line.replace('\n', '').strip()  # newline chars & strip
                              for line in conf_lines]
        conf_lines_cleaned = [line
                              for line in conf_lines_cleaned
                              # empty lines & comments, except for Name line
                              if line and (not line.startswith('#') or line.startswith('# Name'))]

        for line in conf_lines_cleaned:
            # init
            if not in_interface_section:
                if line.startswith("[Interface]"):
                    in_interface_section = True

                else:
                    raise RuntimeError(f"error: parsing interface conf file: '{conf_fp}'")

            # parse interface section
            elif in_interface_section:
                # error: entering unrecognised section
                if line.startswith('['):
                    raise RuntimeError(f"error: unrecognised section: '{line}' in interface file: '{conf_fp}'")

                else:
                    acc = Interface.parse_conf_line(acc, line)

        return Interface.from_dict(cast(types.InterfaceDict, acc))

    def as_ini(self, print_header=True) -> str:
        ini_str: str
        if print_header:
            ini_str = (
                '# Generated by fugue\n'
                '[Interface]\n'
            )

        else:
            ini_str = '[Interface]\n'

        if self.name:
            ini_str += f'# Name = {self.name}\n'

        if self.address:
            ini_str += f'Address = {self.address}\n'

        if self.listen_port:
            ini_str += f'ListenPort = {self.listen_port}\n'

        if self.private_key:
            ini_str += f'PrivateKey = {self.private_key}\n'

        if self.dns:
            ini_str += f"DNS = {', '.join([str(d) for d in self.dns])}\n"

        if self.table:
            ini_str += f'Table = {self.table}\n'

        if self.mtu:
            ini_str += f'MTU = {self.mtu}\n'

        if self.pre_ups:
            for pre_up in self.pre_ups:
                ini_str += f'PreUp = {pre_up}\n'

        if self.post_ups:
            for post_up in self.post_ups:
                ini_str += f'PostUp = {post_up}\n'

        if self.pre_downs:
            for pre_down in self.pre_downs:
                ini_str += f'PreDown = {pre_down}\n'

        if self.post_downs:
            for post_down in self.post_downs:
                ini_str += f'PostDown = {post_down}\n'

        return ini_str

    def full_path(self) -> pathlib.Path:
        # TODO: how do we name the file if there is no name? let's do <public_key>.conf for now
        _basename: str = (
            f'{self.name}.conf'
            if self.name
            else f'{wg.gen_public_key(self.private_key)}.conf'
        )
        return pathlib.Path(config.FUGUE_INTERFACES_ROOT, _basename)

    def persist(self) -> None:
        utils.write_protected(self.full_path(), self.as_ini())

    @staticmethod
    def get_interfaces() -> List['Interface']:
        interfaces: List['Interface'] = []
        conf_fps: List[pathlib.Path] = [pathlib.Path(config.FUGUE_INTERFACES_ROOT, child)
                                        for child in os.listdir(config.FUGUE_INTERFACES_ROOT)
                                        if pathlib.Path(config.FUGUE_INTERFACES_ROOT, child)
                                        and child.endswith('.conf')]

        for conf_fp in conf_fps:
            # TODO: we should try/catch here and report on malformed conf files
            interfaces.append(Interface.from_conf(conf_fp))

        return interfaces

    @staticmethod
    def rm(name: str) -> None:
        conf_fp: pathlib.Path = pathlib.Path(config.FUGUE_INTERFACES_ROOT, f'{name}.conf')

        if not os.path.isfile(conf_fp):
            raise RuntimeError(f'error: interface with name: {name} not found at expected: {conf_fp}')

        os.remove(conf_fp)

        # TODO: Should we remove the interfaces directory if it's become empty?
        utils.rm_dir_if_empty(config.FUGUE_INTERFACES_ROOT)

    @staticmethod
    def from_name(name: str) -> 'Interface':
        interfaces: List['Interface'] = Interface.get_interfaces()

        matching_interfaces: List['Interface'] = [p for p in interfaces if p.name == name]
        if len(matching_interfaces) == 0:
            raise RuntimeError('error: interface with name: {name} not found.')

        elif len(matching_interfaces) > 1:
            raise RuntimeError('error: found multiple interfaces with name: {name}: {matching_interfaces}')

        return matching_interfaces[0]
