import os
import pathlib
from dataclasses import dataclass
from typing import List, Dict, Any, cast

from fugue import config
from fugue.lib import Interface, Peer, utils, types
from fugue.lib.subprocess import systemctl, wg


@dataclass
class Tunnel:
    """
    An interface and list of peers.
    """
    interface: Interface
    peers: List[Peer]

    @staticmethod
    def from_conf(conf_fp: pathlib.Path) -> 'Tunnel':
        with open(conf_fp, 'r') as fd:
            conf_lines = fd.readlines()

        # init vars
        interface: Interface
        peers: List[Peer] = []

        # parsing
        in_peer_section = False
        in_interface_section = False
        # accumulator
        acc: Dict[str, Any] = {}

        conf_lines_cleaned = [line.replace('\n', '').strip()  # newline chars & strip
                              for line in conf_lines]
        conf_lines_cleaned = [line
                              for line in conf_lines_cleaned
                              # empty lines & comments, except for Name line
                              if line and (not line.startswith('#') or line.startswith('# Name'))]

        for line in conf_lines_cleaned:
            # init
            if not in_interface_section and not in_peer_section:
                if line.startswith("[Interface]"):
                    in_interface_section = True

                elif line.startswith("[Peer]"):
                    in_peer_section = True

            # parse an interface section
            elif in_interface_section:
                # error: finding another interface section
                if line.startswith('[Interface]'):
                    raise RuntimeError(f"error: tunnel file: '{conf_fp}' defines multiple interface sections")

                # stop condition: entering [Peer] section
                elif line.startswith('[Peer]'):
                    # build previous interface from accumulator
                    interface = Interface.from_dict(cast(types.InterfaceDict, acc))
                    # reset accumulator
                    acc = {}
                    # update boolean flags
                    in_peer_section = True
                    in_interface_section = False

                # error: unrecognised section
                elif line.startswith('['):
                    raise RuntimeError(f"error: unrecognised section: '{line}' in interface file: '{conf_fp}'")

                else:
                    # update accumulator with [Interface] section line
                    acc = Interface.parse_conf_line(acc, line)

            # parse a peer section
            elif in_peer_section:
                # stop condition: entering an [Interface] section
                if line.startswith('[Interface]'):
                    # build previous peer from accumulator
                    peers.append(Peer.from_dict(cast(types.PeerDict, acc)))
                    # reset accumulator
                    acc = {}
                    # update boolean flags
                    in_peer_section = False
                    in_interface_section = True

                # stop condition: entering [Peer] section
                elif line.startswith('[Peer]'):
                    # build previous peer from accumulator
                    peers.append(Peer.from_dict(cast(types.PeerDict, acc)))
                    # reset accumulator
                    acc = {}
                    # update boolean flags
                    in_peer_section = True
                    in_interface_section = False

                # error: unrecognised section
                elif line.startswith('['):
                    raise RuntimeError(f"error: unrecognised section: '{line}' in interface file: '{conf_fp}'")

                else:
                    # update accumulator with [Peer] section line
                    acc = Peer.parse_conf_line(acc, line)

        if in_interface_section:
            interface = Interface.from_dict(cast(types.InterfaceDict, acc))

        elif in_peer_section:
            peers.append(Peer.from_dict(cast(types.PeerDict, acc)))

        return Tunnel(interface, peers)

    def get_name(self) -> str:
        return self.interface.name or "unnamed"

    def full_path(self) -> pathlib.Path:
        _basename: str = f'{self.get_name()}.conf'
        return pathlib.Path(config.FUGUE_TUNNELS_ROOT, _basename)

    def as_ini(self) -> str:
        ini_str: str = self.interface.as_ini(print_header=False)

        for peer in self.peers:
            ini_str += '\n'
            ini_str += peer.as_ini(print_header=False)

        return ini_str

    def persist(self) -> None:
        utils.write_protected(self.full_path(), self.as_ini())

    def add_peer(self, peer: Peer) -> None:
        self.peers.append(peer)
        # TODO: is materialized? If so, concatenate to file? Or rewrite the file?
        # TODO: is running?

    @staticmethod
    def rm(name: str) -> None:
        conf_fp: pathlib.Path = pathlib.Path(config.FUGUE_TUNNELS_ROOT, f'{name}.conf')

        if not os.path.isfile(conf_fp):
            raise RuntimeError(f'error: tunnel with name: {name} not found at expected: {conf_fp}')

        os.remove(conf_fp)

        # TODO: Should we remove the tunnels directory if it's become empty?
        utils.rm_dir_if_empty(config.FUGUE_TUNNELS_ROOT)

    @staticmethod
    def get_tunnels() -> List['Tunnel']:
        tunnels: List['Tunnel'] = []
        conf_fps: List[pathlib.Path] = [pathlib.Path(config.FUGUE_TUNNELS_ROOT, child)
                                        for child in os.listdir(config.FUGUE_TUNNELS_ROOT)
                                        if pathlib.Path(config.FUGUE_TUNNELS_ROOT, child)
                                        and child.endswith('.conf')]

        for conf_fp in conf_fps:
            # TODO: we should try/catch here and report on malformed conf files
            tunnels.append(Tunnel.from_conf(conf_fp))

        return tunnels

    @staticmethod
    def from_name(name: str) -> 'Tunnel':
        tunnels: List['Tunnel'] = Tunnel.get_tunnels()
        return [tunnel for tunnel in tunnels if tunnel.get_name() == name].pop()

    def is_promoted(self) -> bool:
        # NOTE: A tunnel is considered promoted if there's a symlink to it
        # under /etc/wireguard/, e.g. it can be managed with wg-quick.
        # NOTE: Do we really need this?
        return os.path.islink(pathlib.Path(config.WG_ROOT, f'{self.get_name()}.conf'))

    def promote(self) -> None:
        if not self.is_promoted():
            os.symlink(self.full_path(), pathlib.Path(config.WG_ROOT, f'{self.get_name()}.conf'))

    def demote(self) -> None:
        if self.is_promoted():
            os.unlink(pathlib.Path(config.WG_ROOT, f'{self.get_name()}.conf'))

    def is_enabled(self) -> bool:
        # TODO: Implement me!
        raise NotImplementedError

    def is_running(self) -> bool:
        status: str = wg.status()
        interface_lines: List[str] = [line.strip() for line in status.split('\n')]

        # NOTE: we don't wanna use systemctl to check if a tunnel is up anymore, see todo notes
        # status: str = systemctl.status(f'wg-quick@{self.get_name()}.service')
        # status_lines: List[str] = [line.strip() for line in status.split('\n')]
        # active_line: str = [line for line in status_lines if line.startswith('Active:')].pop()

        return f"interface: {self.get_name()}" in interface_lines

    def start(self):
        if self.is_running():
            return

        if not self.is_promoted():
            self.promote()

        # NOTE: we don't wanna use systemctl to start/stop tunnels, but wg-quick
        # systemctl.start(f"wg-quick@{self.get_name()}.service")
        wg.quick("up", f"{self.get_name()}")

    def stop(self):
        if self.is_running():
            # NOTE: we don't wanna use systemctl to start/stop tunnels, but wg-quick
            # systemctl.stop(f"wg-quick@{self.get_name()}.service")
            wg.quick("down", f"{self.get_name()}")

    def enable(self):
        if not self.is_promoted():
            self.promote()
        systemctl.enable(f"wg-quick@{self.get_name()}.service", now=True)

    def disable(self):
        if not self.is_enabled():
            return

        if not self.is_promoted():
            self.promote()
        systemctl.disable(f"wg-quick@{self.get_name()}.service", now=True)
